% Model of a steady state isothermal plug flow reactor - Gas phase

% Main file

% For more information, see:

% A. De Visscher 
% Lecture Notes in Chemical Engineering Kinetics and Chemical Reactor Design
% Createspace (2013) 

% (C) Alex De Visscher (2013)

% ---------------------------

% data

clear all

W = 1010;            % Catalyst mass (kg)
Wspan = 0:10:W;      % We want output every 10 kg up to W kg
nE0 = 70;            % Initial alkene flow rate (mol/s)
nH0 = 10;            % Initial hydrogen flow rate (mol/s)
nA0 = 0;             % Initial alkane flow rate (mol/s)

k2 = 0.014;          % Rate constant (mol/bar1.5/kg/s)
KE = 0.12;           % Rate constant (1/bar)
KH = 0.025;          % Rate constant (1/bar)
KA = 0.075;          % Rate constant (1/bar)
p = 500000;          % Pressure (Pa)

rhob = 450;          % Bed density of catalyst (kg/m3)
L = 4;               % Bed depth (m)
V = W/rhob;          % Reactor bed volume (m3)
OMEGA = V/L;         % Reactor cross sectional area
ME = 0.084;          % Molar mass of E (kg/mol)
MH = 0.002;          % Molar mass of hydrogen
MA = 0.086;          % Molar mass of A
Temp = 673.15;       % Temperature (K)
dp = 0.006;          % Catalyst particle diameter (m)
mu = 2.5e-4;         % Gas viscosity (Pa s)
phi = 0.4;           % Bed porosity (-)

n0(1) = nE0;         % Running variable for nE is N(1)
n0(2) = nH0;         % Running variable for nH is N(2)
n0(3) = nA0;         % Running variable for nA is N(3)
n0(4) = p;           % Running variable for p is N(4)

% Call of differential equation solver

options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'InitialStep', 0.01);
[W,N] = ode45(@f,Wspan,n0,options,k2,KE,KH,KA,ME,MH,MA,OMEGA,mu,Temp,dp,rhob,phi);

% Output

figure(1)
plot(W,N(:,1:3))
xlabel('Catalyst mass (kg)');
ylabel('Molar flow rate (mol/s)');
figure(2)
plot(W,N(:,4))
xlabel('Catalyst mass (kg)');
ylabel('Pressure (Pa)');
W
N
[s1,s2] = size(W);
conversion = (N(1,2)-N(s1,2))/N(1,2)
